function value = value_default_leg_index(dates, T, coupon, portfolio, index_members, discounts, discounts_inbetween, maturities, LGD_mean,...
                                         ajd_common, y0)
% --------------------------------------------------------------------------------------------------
% Calculate the market value of the payments made by the seller of protection for an index. This
% function is fully vectorized.
% --------------------------------------------------------------------------------------------------
% dates                 ... datenums for which to compute value of fixed leg payments
% T                     ... time to maturity of index
% coupon                ... coupon rate of index
% index_members         ... indicator matrix with index members at each point in time
% discounts             ... discount curves (matching coupon payments, end of period)
% discounts_inbetween   ... discount curves (matching coupon payments, middle of period)
% maturities            ... which maturities to update (indicator vector), default: update all
% LGD_mean              ... mean LGD
% ajd_common            ... dynamics of common factor
% y0                    ... initial value(s) of common factor
% --------------------------------------------------------------------------------------------------
% sample call:
%{
    value_default_leg_index(datenum({'11/20/2006' '11/21/2006'}), [5; 4.9974], repmat(cds_test, 5, 1), [1 1 1 1 1; 1 1 1 1 0], ...
                            exp(-0.05*repmat(0.25:0.25:5, 2, 1)), exp(-0.05*repmat((0.25:0.25:5)-1/8, 2, 1)))
%}
% --------------------------------------------------------------------------------------------------

% Compute expected quarterly portfolio notional loss;
[exp_notional, exp_notional_loss] = expected_index_notional(dates, T, portfolio, index_members, maturities, ajd_common, y0);

% Compute market value of default payments made buy seller of protection for index
value = zeros(length(dates), 1);
for i=1:length(dates)    
    for t=1:ceil(T(i)*4)
        % Add discounted payment to market value of tranche
        value(i) = value(i) + exp_notional_loss(i,t) .* discounts_inbetween(i,t) * LGD_mean;
        
        % Subract accrued premium between last coupon payment date and time of default
        value(i) = value(i) - coupon(i)/1e4/4/2 * (365/360) * discounts_inbetween(i,t) * exp_notional_loss(i,t);
    end
end